<?php

namespace App\Http\Controllers\Guru;

use App\Http\Controllers\Controller;
use App\Models\PeminjamanAlat;
use App\Models\Barang;
use Illuminate\Support\Facades\DB;

class GuruPeminjamanController extends Controller
{
    public function index()
    {
        if (!session('guru_id')) return redirect()->route('login');

        $guruId = session('guru_id');

        $peminjaman = PeminjamanAlat::where(function ($q) use ($guruId) {
                $q->where('peminjam_role', 'siswa')
                  ->where(function ($sub) use ($guruId) {
                      $sub->whereNull('id_guru_persetujuan')
                          ->orWhere('id_guru_persetujuan', $guruId);
                  });
            })
            ->orWhere(function ($q) use ($guruId) {
                $q->where('peminjam_role', 'guru')
                  ->where('peminjam_id', $guruId);
            })
            ->orderBy('created_at', 'desc')
            ->get();

        return view('guru.peminjaman.index', compact('peminjaman'));
    }

    public function detail($id)
    {
        if (!session('guru_id')) return redirect()->route('login');

        $peminjaman = PeminjamanAlat::with('detail.barang')->findOrFail($id);

        if (
            $peminjaman->id_guru_persetujuan &&
            $peminjaman->id_guru_persetujuan != session('guru_id')
        ) {
            abort(403, 'Peminjaman ini sedang ditangani guru lain');
        }

        return view('guru.peminjaman.detail', compact('peminjaman'));
    }

    // ===================== APPROVE =====================
    public function approve($id)
    {
        if (!session('guru_id')) return redirect()->route('login');

        DB::beginTransaction();

        try {
            $peminjaman = PeminjamanAlat::with('detail.barang')->lockForUpdate()->findOrFail($id);

            // Sudah diproses sebelumnya
            if ($peminjaman->status !== 'pending') {
                return back()->with('error', 'Peminjaman sudah diproses');
            }

            // Ditangani guru lain
            if (
                $peminjaman->id_guru_persetujuan &&
                $peminjaman->id_guru_persetujuan != session('guru_id')
            ) {
                return back()->with('error', 'Peminjaman sudah ditangani guru lain');
            }

            // Cek stok
            foreach ($peminjaman->detail as $detail) {
                if ($detail->barang->stok < $detail->jumlah) {
                    DB::rollBack();
                    return back()->with(
                        'error',
                        'Stok barang "' . $detail->barang->nama . '" tidak mencukupi'
                    );
                }
            }

            // Kurangi stok
            foreach ($peminjaman->detail as $detail) {
                $barang = $detail->barang;
                $barang->stok -= $detail->jumlah;
                $barang->save();
            }

            // Update status peminjaman
            $peminjaman->update([
                'status' => 'disetujui',
                'id_guru_persetujuan' => session('guru_id')
            ]);

            DB::commit();

            return back()->with('success', 'Peminjaman berhasil disetujui & stok diperbarui');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', 'Terjadi kesalahan sistem');
        }
    }

    // ===================== REJECT =====================
    public function reject($id)
    {
        if (!session('guru_id')) return redirect()->route('login');

        $peminjaman = PeminjamanAlat::findOrFail($id);

        if ($peminjaman->status !== 'pending') {
            return back()->with('error', 'Peminjaman sudah diproses');
        }

        if (
            $peminjaman->id_guru_persetujuan &&
            $peminjaman->id_guru_persetujuan != session('guru_id')
        ) {
            return back()->with('error', 'Peminjaman sudah ditangani guru lain');
        }

        $peminjaman->update([
            'status' => 'ditolak',
            'id_guru_persetujuan' => session('guru_id')
        ]);

        return back()->with('success', 'Peminjaman berhasil ditolak');
    }
}
